<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use App\SessionPackages;
use App\Card;
use App\Slot;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class UpdateSessionStatusCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sessions:update-status';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update session statuses, mark expired sessions, and activate the next session if applicable.';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        Log::info("Starting session status update...");
        
        // Fetch all sessions with status 'Pending'
        $sessions = SessionPackages::whereIn('status', ['Pending','Expired','Upcoming','Accept'])->get();
        
        foreach ($sessions as $session) {
            // Parse the session date
            $sessionDate = Carbon::parse($session->date, 'Asia/Dubai');
        
            // Find the slot for the session
            $slot = Slot::find($session->time);
        
            if ($slot) {
                // Get the check-in time from the slot
                $checkInTime = Carbon::createFromFormat('H:i:s', $slot->check_in, 'Asia/Dubai');
                $currentTime = Carbon::now('Asia/Dubai');
        
                // Log the session information for debugging
                Log::info("Testing session ID: {$session->id}");
                Log::info("Session date: {$sessionDate->toDateString()}");
                Log::info("Slot check-in time: {$checkInTime->format('H:i:s')}");
                Log::info("Current time: {$currentTime->format('H:i:s')}");
        
                // Determine if the session is expired
                if ($sessionDate->isPast() && !$sessionDate->isToday()) {
                    // If the session date is in the past and not today, mark as Expired
                    Log::info("Session ID {$session->id} marked as Expired (past date).");
                    //$session->update(['status' => 'Expired']);
                } elseif ($sessionDate->isToday() && $currentTime->greaterThanOrEqualTo($checkInTime)) {
                    // If the session date is today and the current time is past the check-in time
                    Log::info("Session ID {$session->id} marked as Expired (past time).");
                    //$session->update(['status' => 'Expired']);
                } else {
                    Log::info("Session ID {$session->id} is not expired.");
                    if($session->status == 'Expired'){
                        $session->update(['status' => 'Pending']);
                    }
                }
            } else {
                Log::warning("No slot found for session ID: {$session->id}");
            }
        }

        // Loop through sessions and check if they start within 2 hours
        foreach ($sessions as $session) {
            $slot = Slot::find($session->time);
            if ($slot) {
                $sessionTime = Carbon::createFromFormat('Y-m-d H:i:s', $session->date . ' ' . $slot->check_in);

                if (Carbon::now('Asia/Dubai')->diffInHours($sessionTime, false) <= 2) {
                    // Mark this session as "Accept"
                    if($session->status != "Expired" && $session->status != "Completed"){
                        $card = Card::find($session->cart_id);

                        if ($card) {
                            if ($card->accept_user_id != null && !in_array($session->status, ['Expired', 'Completed'])) {
                                $session->update(['status' => 'Accept']);
                                Log::info("Session ID {$session->id} marked as Accept.");
                            }
                        } else {
                            Log::warning("Card not found for session ID: {$session->id}, skipping.");
                        }
                    }
                    Log::info("Session ID {$session->id} marked as Accept.");
                }
            }
        }

        // Fetch all sessions with status 'Pending'
        $cards = Card::whereIn('status', ['Pending','Expired','Upcoming','Accept'])->get();
        
        foreach ($cards as $card) {
            // Parse the session date
            $sessionDate = Carbon::parse($card->date, 'Asia/Dubai');
        
            // Find the slot for the session
            $slot = Slot::find($card->slot_id);
        
            if ($slot) {
                // Get the check-in time from the slot
                $checkInTime = Carbon::createFromFormat('H:i:s', $slot->check_in, 'Asia/Dubai');
                $currentTime = Carbon::now('Asia/Dubai');
        
                // Log the session information for debugging
                Log::info("Testing card ID: {$card->id}");
                Log::info("Card date: {$sessionDate->toDateString()}");
                Log::info("Slot check-in time: {$checkInTime->format('H:i:s')}");
                Log::info("Current time: {$currentTime->format('H:i:s')}");
        
                // Determine if the session is expired
                if ($sessionDate->isPast() && !$sessionDate->isToday()) {
                    // If the session date is in the past and not today, mark as Expired
                    Log::info("Session ID {$card->id} marked as Expired (past date).");
                    //$card->update(['status' => 'Expired']);
                } elseif ($sessionDate->isToday() && $currentTime->greaterThanOrEqualTo($checkInTime)) {
                    // If the session date is today and the current time is past the check-in time
                    Log::info("Session ID {$card->id} marked as Expired (past time).");
                    //$card->update(['status' => 'Expired']);
                } else {
                    Log::info("Session ID {$card->id} is not expired.");
                    if($card->status == 'Expired'){
                        $card->update(['status' => 'Pending']);
                    }
                }
            } else {
                Log::warning("No slot found for card ID: {$card->id}");
            }
        }

        Log::info("Session status update complete.");

        return Command::SUCCESS;
    }
}
